-- BDE-Range ermitteln, welche Stempelungen zu meinem Input-Datum gehören
CREATE OR REPLACE FUNCTION TPersonal.bdep__records__by__individwt__get (
      _minr       integer,
      _date_start date,
      _date_end   date DEFAULT null
  ) RETURNS tsrange AS $$
    SELECT
      CASE
          WHEN min( bd_anf_rund ) IS null THEN null
          ELSE tsrange( min( bd_anf_rund ), max( bd_anf_rund ) + INTERVAL '12 hours' )
      END
    FROM bdep
    WHERE bd_minr = _minr
      -- timestamp_to_date(bd_anf) geht über Index, daher kein CAST.
      AND timestamp_to_date(bd_anf) BETWEEN _date_start AND coalesce( _date_end, _date_start )
      AND Tpersonal.bdep__individwt__isfirstofday( in_bdanf => bd_anf_rund, in_bdminr => bd_minr )
    ;
  $$ LANGUAGE sql STABLE;    
--

--drop FUNCTION lohnarten__bereits_verbucht ( timestamp, timestamp, boolean,time,time,time,boolean,integer,varchar)
-- Prüfen, ob der Zeitraum bereits verbucht wurde und damit übersprungen werden kann
CREATE OR REPLACE FUNCTION TPersonal.bdep__lohnarten__zeitraum_bereits_in_anderer_lohnart(
    _bdanf           timestamp,
    _bdend           timestamp,
    _bdschichtbeginn boolean,
    _startzeit       time,
    _endezeit        time,
    _loaendtime       time,
    _endkarenz       boolean,
    _loaid            integer,
    _entgeltgruppe   varchar  
  ) RETURNS boolean AS $$
  BEGIN
  -- wenn Beginn einer neuen Schicht, aber in "alter" Zeit.
  -- Start-Spread prüfen
    IF  _bdanf::time < _endezeit 
        AND EXISTS (
          SELECT true
          FROM lohnarten_zeiten 
            JOIN lohnarten on loaz_laid = loa_id
          WHERE loa_id != _loaid
            AND _bdschichtbeginn IS true
            AND TSystem.ENUM_GetValue( loa_entgeltgruppen, _entgeltgruppe ) 
            AND _bdanf::time BETWEEN loaz_starttime_spread AND loaz_starttime
            
          )
    THEN
      RETURN true;
    -- Schichtende überschritten, aber nur um Spread-Grenze
    ELSIF
      _bdend::time > _startzeit 
      AND EXISTS (
          SELECT true                                                                                
          FROM lohnarten_zeiten 
            JOIN lohnarten ON loaz_laid = loa_id
          WHERE loa_id != _loaid
            AND _bdschichtbeginn IS false
            AND TSystem.ENUM_GetValue( loa_entgeltgruppen, _entgeltgruppe ) 
            AND _bdend::time BETWEEN loaz_endtime AND loaz_endtime_spread 
            )
    THEN
       RETURN true;
    --
    ELSIF
      _endkarenz IS true
      AND _bdanf >
          ( _bdanf::date
            + CASE WHEN (_loaendtime::time = '00:00') THEN '24:00'
                   ELSE _loaendtime::time
              END
          )
    THEN
        RETURN true; 
    END IF;
  RETURN false;
END $$ LANGUAGE plpgsql STABLE;

-- Ermittlung des tatsächlichen Start/Endzeitpunktes der Lohnart
CREATE OR REPLACE FUNCTION TPersonal.bdep__lohnarten_startendzeit__get(
  _bdanf           timestamp without time zone,
  _bdend           timestamp without time zone,
  _startzeit       time without time zone,
  _endezeit        time without time zone,
  OUT _starttime     timestamp without time zone,
  OUT _endtime       timestamp without time zone
) AS $$
DECLARE
  end_timestamp   timestamp;
  start_timestamp timestamp;
BEGIN
  
  -- Verhindern von Range-Problemen (lower must be smaller than upper bound)
  IF _startzeit = '00:00' AND _bdanf::date < _bdend::date THEN
      -- Startzeitpunkt heute, Ende morgen -> um an den Startzeitpunkt 00:00 als LA_TIME hinzuzufügen, muss 1 Tag draufaddiert werden.
      start_timestamp := _bdanf::date + _startzeit + INTERVAL '1 days';
  
  -- neuer Zweig #18650 ~ vergessen waren die Fälle 'nur 1 Stempelsatz Nachtschicht' sowie '2 Stempelsätze Nachtschicht, Pause vor Mitternacht' 
  ELSIF _startzeit <> '00:00' AND _bdanf::date < _bdend::date AND _bdend::date + _startzeit < _bdend THEN 
      -- Startzeitpunkt heute, Ende morgen UND Lohnartzeitraum liegt schon auf dem nächsten Tag -> um an den Startzeitpunkt 00:00 als LA_TIME hinzuzufügen, muss 1 Tag draufaddiert werden.
      start_timestamp := _bdanf::date + _startzeit + INTERVAL '1 days';                  
  --
  ELSIF _endezeit = '00:00' AND _bdanf::date = _bdend::date THEN
      -- Wenn Start und Ende-Tag gleich sind, der nach Lohnart ermittelte Endezeitpunkt auf 00:00 liegt, muss 1 Tag auf den Endezeitraum draufrechnen.
      start_timestamp := _bdanf::date + _startzeit;
      end_timestamp :=   _bdend::date + _endezeit + INTERVAL '1 days';
  --
  ELSE
      -- Nichts von alledem - Startzeit ist Datum Anfangsstempelung + Lohnart-Startzeit
      start_timestamp := _bdanf::date + _startzeit;
  END IF;
  
  -- In keinen der oberen Fälle reingekommmen -> äquivalent zu Startzeit start_timestamp berechnen.
  IF end_timestamp IS null THEN
      end_timestamp := _bdend::date + _endezeit; 
  END IF;
  
  -- Hier werden nicht die Summen (Intersects) benötigt sondern der kleinste sowie größte Zeitpunkt, wo sich Stempelung und Lohnarten-Zeiten überlappen.
  _starttime := lower( tsrange( _bdanf, _bdend ) * tsrange( start_timestamp, end_timestamp ))::timestamp;
  _endtime   := upper( tsrange( _bdanf, _bdend ) * tsrange( start_timestamp, end_timestamp ))::timestamp;
  
  
END $$ LANGUAGE plpgsql STABLE;


-- Hauptfunktion zur Ermittlung der Lohnarten 
-- Legt temporäre Tabelle an
-- Aufteilung der BDE-Stempelungen
-- Abzug von Mindest- und Raucherpausen
CREATE OR REPLACE FUNCTION tpersonal.bdep__lohnarten_zuschlaege__get(
      _minr integer,
      _date_start date,
      _date_end date = null,
      _optimierung boolean = false,
      _debug boolean = false
  ) RETURNS void AS $$
  BEGIN

      DROP TABLE IF EXISTS tmp_time_tab;
      -- zum Aufsplitten nach gebuchten Zeiten notwendig
      CREATE TEMPORARY TABLE IF NOT EXISTS tmp_time_tab (
        ttt_id         serial PRIMARY KEY,
        minr           integer NOT null,        -- Mitarbeiter
        schichtbeginn  boolean,                 -- handelt es sich um eine initialstempelung am Arbeitstag
        bd_anf_rund    timestamp,               -- Stempel-Anfang (hauptsächlich zum debuggen)
        bd_end_rund    timestamp,               -- Stempel-Ende  (hauptsächlich zum debuggen)
        loa_anf        timestamp,               -- Startzeitpunkt der Stempelung für die aktuelle Lohnart
        loa_end        timestamp,               -- Endezeitpunkt der Stempelung für die aktuelle Lohnart
        loa_bez        varchar,                 -- Lohnart
        loa_nr         integer NOT null,        -- Lohnartnummer
        sum            numeric(12,4),           -- Zeitdifferenz zwischen la_end und la_anf (abzüglich eventuell nicht verbuchter Mindestpausen!!)
        abrechdatum    timestamp,               -- Abrechnungszeitpunkt
        quelle         varchar(1) DEFAULT 'B',  -- Datenquelle: B = Basisfunktionalität, O = Optimierung        
        min_exceed_sum numeric(12,4),           -- neu in #19032 Zeitdifferenz Über- bzw. Unterzeit ~ positiv = Überstunde
        werktag        date                     -- neu in #19032 individueller Werktag --> auch in TPersonal.bdep__lohnarten__split
      );

      -- Aufteilung der Stempeldaten in die jeweiligen Lohnarten
      PERFORM TPersonal.bdep__lohnarten__split( 
          _minr,
          _date_start,
          _date_end
      );

      -- Abzug der unterschrittenen Mindestpausen
      PERFORM TPersonal.bdep__lohnarten__abzug_fehlende_minpause( 
          _minr,
          _date_start,
          _date_end
      );

      -- Abzug der Raucherpausen
      PERFORM TPersonal.bdep__lohnarten__abzug_fehlende_rpause( 
          _minr,
          _date_start,
          _date_end
      );
      
      -- #19032 muss vor Schreiben der Summe passieren ~ weil Bezug zu el_status: 1 = Summe pro Monat / 0 = Tagesgenau
      -- Optimierung schreibt immer in tmp-table, und in den Export werden nur die Gesamtsummen übernommen
      IF _optimierung then          
          PERFORM TPersonal.bdep__lohnarten__optimierung(
               _minr,
               _date_start,
               _date_end,
               true -- bewusst nicht _debug ~ Optimierung schreibt immer in tmp-table
          );
      END IF;

      -- Gesamtsummen in Export-Tabelle schreiben, falls das _debug-Flag nicht gesetz ist        
      IF NOT _debug THEN
          INSERT INTO exportLohn ( el_abrechdat, el_minr,  el_lohnart, el_anzStunden, el_loa_zusatz, el_status)
          SELECT coalesce( _date_end, _Date_start ),
               _minr,
               loa_nr,
               sum( sum ),
               1,
               1
          FROM tmp_time_tab
          WHERE loa_nr > 0 -- Hinweis: negative Lohnarten sorgen dafür das optimiert werden kann LA -1000 beispw. wird nicht in den Export gegeben, da Where >0 
          GROUP BY 1, 2, 3;
      END IF;        

      IF NOT _debug THEN
        DROP TABLE IF EXISTS tmp_time_tab;
      END IF;
      
  END $$ LANGUAGE plpgsql VOLATILE;
--  

-- Debug-Funktion zum Auslesen der ermittelten Lohnarten-ZUordnungen (vor Optimierung)
CREATE OR REPLACE FUNCTION TPersonal.bdep__lohnarten_zuschlaege__get_debug( 
      _minr           integer,
      _date_start     date,
      _date_end       date DEFAULT null,
      _optimierung    boolean DEFAULT FALSE
  ) RETURNS TABLE (
      ttt_id          integer,
      loa_nr          integer,
      loa_bez         varchar,
      minr            integer,
      bd_firstofday   boolean,
      bd_anf          timestamp,
      bd_end          timestamp,
      loa_anf         timestamp,
      loa_end         timestamp,
      sum             numeric(12,4),
      abrechdatum     timestamp,          -- Abrechnungszeitpunkt
      quelle          varchar(1),
      min_exceed_sum  numeric(12,4),
      werktag         date      
  ) AS $$
  BEGIN

      PERFORM TPersonal.bdep__lohnarten_zuschlaege__get( 
          _minr,
          _date_start,
          _date_end,
          _optimierung ,
          _debug => true
      );

      RETURN QUERY
      SELECT tmp_time_tab.ttt_id,
             tmp_time_tab.loa_nr,
             tmp_time_tab.loa_bez,
             tmp_time_tab.minr,
             tmp_time_tab.schichtbeginn,
             tmp_time_tab.bd_anf_rund,
             tmp_time_tab.bd_end_rund,
             tmp_time_tab.loa_anf,
             tmp_time_tab.loa_end,
             tmp_time_tab.sum,
             tmp_time_tab.abrechdatum,
             tmp_time_tab.quelle,
             tmp_time_tab.min_exceed_sum,
             tmp_time_tab.werktag  
      FROM tmp_time_tab
      ORDER by tmp_time_tab.bd_anf_rund, tmp_time_tab.loa_anf;
    
  END $$ language plpgsql;
--  

-- Funktion zum Splitten der Buchungszeiten entsprechend der hinterlegten Lohnarten
-- größere Überarbeitung 17365, 19032
CREATE OR REPLACE FUNCTION TPersonal.bdep__lohnarten__split(
      _minr       integer,
      _date_start date,
      _date_end   date = null
  ) RETURNS void AS $$
  DECLARE
      rec_bdep          record;
      rec_la            record;
      
      loa_anf           timestamp;
      loa_end           timestamp;
      
      end_timestamp     timestamp;
      start_timestamp   timestamp;
      
      sum_intersect     numeric(12,4);
      
      lohnart_bez       varchar(100);
      lohnart_nr        integer;      
      
  BEGIN
      ---RAISE NOTICE '>> 1_ Ausführung: TPersonal.bdep__lohnarten__split';
      --  Über alle Stempelungen im angefragten Zeitraum
      FOR rec_bdep IN
          SELECT
            bd_anf_rund,
            bd_end_rund,
            bd_individwt_mpl_date, --MF: neu
            Tpersonal.bdep__individwt__isfirstofday( in_bdanf => bd_anf_rund, in_bdminr => _minr ) AS schichtbeginn
          FROM TPersonal.bdep__records__by__individwt__get( _minr, _date_start, _date_end )
            JOIN bdep ON bd_minr = _minr
          WHERE bd_anf <@ bdep__records__by__individwt__get
          ORDER BY bd_anf
      LOOP
          
          FOR rec_la IN
              SELECT loam_tpl_name,
                loa_id, loa_nr, loa_bez, 
                loaz_starttime, loaz_endtime,  
                pers_entgelt, mpl_tpl_name,               
                ( mpl_saldo + mpl_absaldo - TPersonal.bdep__raucherpausen_summe__get( mpl_minr, mpl_date, mpl_date ) ) - mpl_min as mpl_min_exceed_sum -- MF #19032                                                                           
              FROM lohnarten
                JOIN lohnarten_zeiten ON loaz_laid = loa_id
                JOIN personal ON TSystem.ENUM_GetValue( loa_entgeltgruppen, pers_entgelt ) -- nur die zur richtigen Lohnart, im Personalmodul hinterlegt, welche Zuschlagsgruppe
                JOIN llv ON ll_ad_krz = pers_krz
                JOIN mitpln ON mpl_minr = ll_minr AND mpl_date = rec_bdep.bd_individwt_mpl_date -- 17365 Tagesplanbezug
                JOIN lohnmodelle ON loaz_id = loam_loaz_id AND ( mpl_tpl_name = loam_tpl_name ) -- 17365 Tagesplanbezug
              WHERE
                ll_minr = _minr
                AND -- dieser Teil in #19032 angepasst : Am Feiertag wird nur der Feiertagszuschlag gewertet
                (  
                   (  -- Es ist kein Feiertag
                      CASE day_of_week( rec_bdep.bd_anf_rund ) -- welcher Wochentag liegt vor
                        WHEN 1 THEN loaz_is_monday
                        WHEN 2 THEN loaz_is_tuesday
                        WHEN 3 THEN loaz_is_wednesday
                        WHEN 4 THEN loaz_is_thursday
                        WHEN 5 THEN loaz_is_friday
                        WHEN 6 THEN loaz_is_saturday
                        WHEN 0 THEN loaz_is_sunday
                      END
                      AND NOT loaz_is_holiday
                      AND NOT EXISTS ( SELECT true FROM feiertag WHERE ft_date = rec_bdep.bd_anf_rund::date )
                   )
                   OR 
                   (  -- oder es ist Feiertag 
                      loaz_is_holiday
                      AND EXISTS ( SELECT true FROM feiertag WHERE ft_date = rec_bdep.bd_anf_rund::date )
                   )
                )                                      
              ORDER BY loaz_starttime
                        
          LOOP
              -- Überschneidung zwischen Stempel-Zeitraum und Lohnart-Zeiten ermitteln              
              sum_intersect :=
                tsystem.timestamps__intervals__intersect(
                    timestamp_start => rec_bdep.bd_anf_rund,
                    timestamp_end   => rec_bdep.bd_end_rund,
                    interval_start  => rec_la.loaz_starttime, 
                    interval_end    => rec_la.loaz_endtime  
                )
              ; 
              
              ---RAISE NOTICE '3_ Loop 2 Überschneidung zwischen Stempel-zeitraum und Lohnart-Zeiten ermitteln:  %', sum_intersect;
              
              -- Keine Überschneidung              
              IF sum_intersect = 0 THEN
                  ---RAISE NOTICE '4_ Keine Überschneidung zwischen Stempel-zeitraum und Lohnart-Zeiten: %', rec_la.loa_bez;
                  CONTINUE;                  
              END IF;                                        
              
              SELECT  _starttime, _endtime
              INTO    loa_anf,    loa_end
              FROM  TPersonal.bdep__lohnarten_startendzeit__get(
                        _bdanf       => rec_bdep.bd_anf_rund,
                        _bdend       => rec_bdep.bd_end_rund,
                        _startzeit   => rec_la.loaz_starttime, 
                        _endezeit    => rec_la.loaz_endtime  
                    )
              ;
              ---RAISE NOTICE '5_ _bdanf = %, _bdend = %, _startzeit = %, _endezeit = %, Lohnart = %', rec_bdep.bd_anf_rund, rec_bdep.bd_end_rund, rec_la.loaz_starttime, rec_la.loaz_endtime, rec_la.loa_bez;                                  
              
              INSERT INTO tmp_time_tab (minr, schichtbeginn,          bd_anf_rund ,         bd_end_rund,          loa_anf, loa_end, loa_bez,        loa_nr,        sum,           min_exceed_sum,            werktag)-- Erweiterung #19032
              VALUES                   (_minr,rec_bdep.schichtbeginn, rec_bdep.bd_anf_rund, rec_bdep.bd_end_rund, loa_anf, loa_end, rec_la.loa_bez, rec_la.loa_nr, sum_intersect, rec_la.mpl_min_exceed_sum, rec_bdep.bd_individwt_mpl_date);              
              ---RAISE NOTICE '6_ Insert 1 done; [bd_anf_rund= % | bd_end_rund= % | loa_anf= % | loa_end= % | loa_bez= % ]', rec_bdep.bd_anf_rund, rec_bdep.bd_end_rund, loa_anf, loa_end, rec_la.loa_bez;                                             
              
          END LOOP; -- Lohnarten
          
      END LOOP; -- BDE-Stempelungen
      
      RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--

-- Funktion führt auf die erste gefunde Lohnart den Abzug aller Pausen aus: Tagesplan-Pausen, feste Pausen, Mindestpausen
-- Die Pause soll in der (für den Mitarbeiter) schlechteren Lohnart abgezogen werden, damit die bessere Lohnart voll bezahlt wird
-- Fälle Mindestpausenabzug für Tagesplan-Lohnart-Varianten
  --  Beginnen in Frühschicht >> gibt keine besseren Zuschläge
  --  Beginnen in Spätschicht >> gibt keine besseren Zuschläge
  --  Beginnen in Nachtschicht >> von 24:00 bis 04:00 die bestbezahlten Zuschläge ~ Pause am besten in der ersten gefundenen Lohnart abziehen
  --  Für alle Tagesplan-Möglichkeiten gilt also, immer in der ersten gefundenen Lohnart abziehen
-- Beachten: 
  -- Leider ist das Merkmal Schichtbeginn aus TPersonal.bdep__lohnarten__split nicht verwendbar, wenn es sich nur um eine 1 Stempelung handelt: dann ist alles Schichtbeginn
-- Es gibt Pausen durch Mindestpause global 1), feste Pause 2), und Tagesplanpause direkt zum Tagesplan 3)
--- 1) Abzug durch bdab_stu
--- 2) Abzug durch bd_gleitpause 
--- 3) Abzug durch bdab_stu
-- die Berücksichtigung von mpl_absaldo wie in Vorgängerversionen ist hier nicht zielführend, denn mpl_absaldo enthält die summierten Abzüge
CREATE OR REPLACE FUNCTION TPersonal.bdep__lohnarten__abzug_fehlende_minpause(
      _minr        integer,
      _date_start  date,
      _date_end    date = null
  ) RETURNS void AS $$
  DECLARE
      rec_pause       record;
      rec_la          record;
  BEGIN
    -- Für alle Pausen, die wir in diesem Zeitraum finden: Tagesplan-Pausen, feste Pausen, Mindestpausen
    FOR rec_pause IN
        SELECT DISTINCT mpl_date, mpl_absaldo, bdab_stu, bd_gleitpause
        FROM mitpln        
          LEFT JOIN bdep       ON mpl_minr = bd_minr AND mpl_date = bd_individwt_mpl_date AND bd_gleitpause IS NOT NULL --feste Pausen der Tagespläne 
          LEFT JOIN bdepab     ON mpl_minr = bdab_minr AND mpl_date = bdab_anf AND bdab_aus_id = 101 --bereits gesetzte Mindestpause, es kann immer nur 1 Mindestpause geben ist systemseitig sichergestellt
        WHERE
          mpl_minr = _minr
          AND ( mpl_absaldo < 0  -- Nur wenn es wirklich noch einen notwendigen Abzug gibt; globale Mindestpause in mpl_absaldo enthalten
               OR 
               bdab_stu < 0      -- Abzug Mindestpause bzw. Tagesplanpause direkt zum Tagesplan
               OR
               bd_gleitpause > 0 -- feste Pause des Tagesplans
              )               
          AND mpl_date BETWEEN _date_start AND coalesce( _date_end, _date_start )  -- nur für den betroffenen Zeitraum
        ORDER BY mpl_date
    LOOP
        -- Für alle Lohnarten, die wir in diesem Zeitraum finden ~ mehrere Tage
        FOR rec_la IN
            SELECT *
            FROM tmp_time_tab
            WHERE
                loa_anf::date = rec_pause.mpl_date
                AND ( bd_anf_rund::date = rec_pause.mpl_date 
                      AND schichtbeginn ) --sonst treffen wir bei Nachtschicht wieder den Vortag bei mehreren Tagen Input
                AND minr = _minr
            ORDER by tmp_time_tab.bd_anf_rund, tmp_time_tab.loa_anf
            LIMIT 1 -- Bedingung: Abzug immer in der ersten gefundenen Lohnart, siehe Beschreibung vor dem Create
        LOOP  
            -- gewertete Zeit in der temporären Tabelle mit dem Abzug der Mindestpause versehen
            UPDATE tmp_time_tab SET
                sum = sum + COALESCE(rec_pause.bdab_stu, 0) - COALESCE(rec_pause.bd_gleitpause, 0)                 
            WHERE 
                ttt_id = rec_la.ttt_id                            
            ;               
            
        END LOOP;
            
    END LOOP;
    
    RETURN;
  END $$ LANGUAGE plpgsql;  
--

-- Abzug noch nicht berücksichtigter Raucherpausen #14030, #18800
-- Ansatz ist zuerst über die Raucherpausen zu gehen, weil dann die Anzahl der Schleifen bzw Anzahl der RP pro Arbeitstag feststehen
  -- den frühesten und spätesten Datensatz bdep zum individuellen Werktag per Lateral und bdepab auf diese Maximal-zeiten Joinen
-- Im zweiten Loop werden über den Zeitraum nur die passenden Lohnarten zu den Raucherpausen geholt und die Temporäre Lohnarten-Tabelle aktualisert
CREATE OR REPLACE FUNCTION tpersonal.bdep__lohnarten__abzug_fehlende_rpause(
      _minr       integer,
      _date_start date,
      _date_end   date = null
  ) RETURNS void AS $$ 
  DECLARE
      rec_pause     record;
      rec_la        record;
      sum_intersect numeric(12,2);
  BEGIN
    -- Raucherpausen und deren Beginn- und Endezeiten im passenden Zeitraum holen    
    FOR rec_pause IN
        SELECT DISTINCT 
            mpl_date , bdab_anf, bdab_anft, bdab_end, bdab_endt, ( bdab_anf || ' ' ||  bdab_anft )::timestamp AS bdab_anfstamp
        FROM mitpln     
            JOIN LATERAL ( SELECT min(bd_anf) AS min_bd_anf, max(bd_end) AS max_bd_end
                 FROM bdep   
                 WHERE bd_minr = _minr AND bd_individwt_mpl_date = mpl_date
                 ) AS lat ON true 
            JOIN bdepab ON bdab_minr = mpl_minr 
                AND bdab_aus_id = 103              
                AND 
                  ( bdab_anf || ' ' ||  bdab_anft )::timestamp
                    BETWEEN min_bd_anf AND max_bd_end
        WHERE 
            mpl_date BETWEEN _date_start AND _date_end 
            AND mpl_minr = _minr 
    LOOP
        -- Für alle Lohnarten-Stempelungen, in denen es Pausen gab
        FOR rec_la IN
            SELECT *
            FROM tmp_time_tab
            WHERE
              rec_pause.bdab_anfstamp BETWEEN loa_anf AND loa_end                         
              AND minr = _minr
            ORDER by tmp_time_tab.bd_anf_rund, tmp_time_tab.loa_anf
        LOOP
            -- es wird die Pausenlänge ermittelt
            sum_intersect :=
              TPersonal.bdep__raucherpausen_summe__get(
                  _minr,
                  rec_la.loa_anf::date,
                  rec_la.loa_end::date,
                  rec_pause.bdab_anft, -- siehe Kommentar im Runner https://ci.prodat-sql.de/sources/tests/suite/10/runner/354#teststep-14215 
                  rec_pause.bdab_endt
              )
            ;
            -- Die Pausenzeit wird von der Lohnartzeit abgezogen
            IF sum_intersect > 0 THEN
                UPDATE tmp_time_tab SET
                  sum = sum - sum_intersect
                WHERE ttt_id = rec_la.ttt_id;                
                -- Zur Sicherheit verlassen wir den Loop, weil die Pause jetzt bereits abgezogen ist
                Exit;                
            END IF;        
        END LOOP; --rec_la
    END LOOP;--rec_pause
    
    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;
--  

-- Ermittlung der Raucherpausen im angegebenen Zeitraum für die jeweilige übergebene Lohnart
CREATE OR REPLACE FUNCTION tpersonal.bdep__lohnarten__rpause__by_lohnart__get(
      _minr           integer,
      _date_start     date,
      _date_end       date ,
      _time_start     time,
      _time_end       time,
      _lohnarten      integer[],
      _check_end      boolean = false
  ) RETURNS numeric AS $$ 
  DECLARE
      raucherpause numeric(12,2) := 0;
      rec_la       record;
  BEGIN
    -- Für alle Lohnarten-Stempelungen in diesem Zeitraum
    FOR rec_la IN
        SELECT DISTINCT
          loa_anf::date,
          CASE
            WHEN 
                    _check_end
                AND loa_end::date = loa_anf::date
            THEN
                loa_end::date + interval '1 day'
            ELSE
                loa_end::date
          END AS loa_end
          
        FROM tmp_time_tab
        WHERE
              loa_nr = ANY( _lohnarten )
          AND minr = _minr
          AND loa_anf::date >= _date_start
              -- abhängig von Anfangs oder Ende-betreffende Raucherpausen
          AND (
                (
                  _check_end
                  AND 
                  loa_end BETWEEN loa_anf::date + _time_start AND loa_anf::date + interval '1 day' + _time_end
                )
                OR                     
                (
                  NOT _check_end 
                  AND 
                  loa_anf BETWEEN loa_anf::date + _time_start AND loa_anf::date +_time_end
                )
          )
          AND loa_end::date <= _date_end
    LOOP
        raucherpause :=
            raucherpause
          + TPersonal.bdep__raucherpausen_summe__get(
                _minr,
                rec_la.loa_anf,
                rec_la.loa_end::date,
                _time_start,
                _time_end
            )
        ;
    END LOOP;
    
    RETURN raucherpause;
  END $$ LANGUAGE plpgsql VOLATILE;
--

-- Die Lohnart ist über die Entgeltgruppe aus Lohnart-Tabelle und des Personalmoduls passend zum Mitarbeiter
CREATE OR REPLACE FUNCTION TPersonal.lohnarten__entgeltgruppe__minr__check(
      _minr         integer,
      _loa_nr       integer
  ) RETURNS boolean AS $$
  BEGIN
      RETURN 
        EXISTS(
            SELECT true
            FROM lohnarten
              JOIN personal ON TSystem.ENUM_GetValue( loa_entgeltgruppen, pers_entgelt )
              JOIN llv ON ll_ad_krz = pers_krz
            WHERE ll_minr = _minr
              AND loa_nr  = _loa_nr
        );
  END $$ LANGUAGE plpgsql;
--


-- in #19032 Anpassung bzgl el_status ~ weil Bezug zu tpersonal.bdep__lohnarten_zuschlaege__get: el_status 1 = Summe pro Monat / 0 = Tagesgenau 
CREATE OR REPLACE FUNCTION tpersonal.bdep__lohnarten__optimierung__write(
      _abrechdat timestamp,
      _minr integer,
      _loa_nr integer,
      _anz_stunden numeric(12,4),
      _debug boolean 
  ) RETURNS void AS $$
  DECLARE
    _status    integer;
  BEGIN

  -- schreibt die Optimierungsdaten des Lohnartenexport in Abhängikeit vom Debug-Status
 
    IF _debug THEN

        INSERT INTO tmp_time_tab 
          (  minr,  loa_nr,          sum,  abrechdatum, quelle, loa_bez )
        VALUES                   
          ( _minr, _loa_nr, _anz_stunden,   _abrechdat,    'O', (SELECT loa_bez FROM lohnarten WHERE loa_nr = _loa_nr LIMIT 1 ) );

    ELSE
      
        -- Berücksichtigung Status im Export: el_status 1 = Summe pro Monat / 0 = Tagesgenau 
        IF _anz_stunden >= 0 THEN
          _status:=1;
        ELSE 
          _status:=0; -- Das sind die taggenauen Abzüge
        END IF;
      
        INSERT INTO exportLohn
          ( el_abrechdat, el_minr, el_lohnart, el_anzStunden, el_loa_zusatz, el_status )
        VALUES
          (   _abrechdat,   _minr,    _loa_nr,  _anz_stunden,             1,   _status );      

    END IF;

  END $$ LANGUAGE plpgsql VOLATILE;
-- 

-- Diese Funktion gibt die Zeibereiche der Optimierungslohnart aus #19313
-- ist beschränkt für die Angabe von zwei dynamischen Lohnartzeitbereichen
CREATE OR REPLACE FUNCTION TPersonal.bdep__lohnarten__optimierung__lohnart__get(
    _interval_id     integer = null, -- entspricht row number / null Angabe wenn nur die Lohnart-Nummer ausgegeben werden soll

    OUT int_start    time,
    OUT int_end      time,
    OUT loaNr        integer,  -- für Verwendung: get Optimierungslohnart, keine Setting-Abfrage erforderlich
    OUT isValid      boolean, 
    OUT errorString  text      -- Fehlerbehnadlung
  ) RETURNS record AS $$
  DECLARE
      rownums integer;
  BEGIN

      SELECT loaz_starttime, loaz_endtime, loa_nr, count(loaz_id) over ()
      INTO   int_start,      int_end,      loaNr,  rownums
      FROM lohnarten_zeiten
      JOIN lohnarten ON loaz_laid = loa_id
      WHERE loa_nr = 1530 -- Optmierungslohnart, kann auch durch Setting ersetzt werden
      ORDER BY
        CASE WHEN (_interval_id = 1) THEN loaz_starttime END DESC,
        CASE WHEN (_interval_id = 2) THEN loaz_starttime END ASC
      LIMIT 1
      ;
      
      isValid := TRUE;
      -- Fehlerbehandlung nur notwendig, wenn die Optimierung aktiv ist
      -- Die Optimierungsfunktion benötigt genau 2 Zeilen mit eingetragenen Zeitbereichen
      IF ( TSystem.Settings__GetBool( 'BDE.Lohnart.KB_MA_LStSV_SpaetNacht_Optimierung' ) ) THEN

          IF (   int_start IS NULL
              OR int_end   IS NULL) THEN
             isValid := False;
             errorString := lang_text(25543); -- start or endtime missing
          END IF;
          
          IF ( rownums IS NULL or rownums <> 2 ) THEN
             isValid := False;
             errorString := lang_text(25542); -- exactly 2 rows requiered            
          END IF;

      END IF;

  END $$ LANGUAGE plpgsql STABLE;
--   

-- Lohnartenoptimierungsfunktion zugunsten der Arbeitsnehmer
-- Grundsätzlich: Die Optimierung bedeutet, dass Mitarbeitende ihre Überstunden von STEUERPFLICHTIGEN Zuschlägen zu STEUERFREIEN verbessern können.
-- Voraussetzung sind auf die Monatssumme gesehen: IST-Stunden > SOLL-Stunden. Überstunden können nur einmal mit Zuschlägen ausgezahlt werden.
-- Sollzeit nicht erreicht (stukosaldo < 0) und Überstundenauszahlung, dann wird nichts optimiert, aber die Überstunden ausgezahlt 
-- in #19032 ohne Berücksichtigung der Vorgängerversion neuerstellt
CREATE OR REPLACE FUNCTION TPersonal.bdep__lohnarten__optimierung(
      _minr         integer,
      _date_start   date,
      _date_end     date = null,
      _debug        boolean = false
  ) RETURNS void AS $$
  DECLARE

      urlaub_ausz             numeric(6,2);
      ueberstunden_ausz       numeric(6,2);      
      stukosaldo              numeric(6,2);
      
      rec_tmp_tab             record;
      
      ueberstunden_wt         numeric(6,2);       -- Überstunden des Werktags
      ueberstunden_wt_sum     numeric(6,2) := 0;  -- Überstunden summiert
      werktag_last            date := Null;       -- letzter Werktag 
      sum                     numeric(6,2);       -- Variable für Lohnart-Betrag

      la_opt                  integer;            -- Optimierungslohnart #19313
      la_opt_1_start          time;
      la_opt_1_end            time;
      la_opt_2_start          time;
      la_opt_2_end            time;

  BEGIN
    
    ------------------------------------------- Urlaubsauszahlung -------------------------------        
    
    /* Auskommentiert, da nicht korrekt
       Das ist soweit richtig, aber die Funktion tpersonal.bdep__lohnarten__optimierung__write kann bisher nur Stunden
       lässt sich aber einfach erweitern
    -- Ermittlung des ausgezahlten Urlaubs
    SELECT  sa_urlaub
    INTO    urlaub_ausz
    FROM public.stundauszahl
    WHERE
          sa_minr = _minr
      AND sa_date BETWEEN _date_start AND _date_end
      AND sa_type = 'holidays'
    ;      
    
    IF urlaub_ausz > 0 THEN
      PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
        coalesce( _date_end, _date_start ), 
        _minr, 
        4060, --fix Lohnart für Urlaubsauszahlung nach Kundenvorgabe
        urlaub_ausz,  
        _debug 
      );
    END IF; */
    
    ------------------------------------------- Überstundenauszahlung -------------------------------
     
    -- Ermittlung Stundenkonto-Saldo ~ diese Funktion existiert nur hier   
    stukosaldo := 
        TPersonal.llv__mitpln__stundenkonto_saldo__calc(minr, date_trunc('month', werktag)::date, (date_trunc('month', werktag) + interval '1 month' - interval '1 day')::date, true )
        FROM tmp_time_tab
        WHERE              
          minr = _minr
          AND bd_anf_rund <@ TPersonal.bdep__records__by__individwt__get( _minr, _date_start, _date_end )
        LIMIT 1
    ;
    ---Raise notice 'stukosaldo = %', stukosaldo;
    
    -- Ermittlung der ausgezahlten Überstunden
    SELECT  sa_stunden 
    INTO    ueberstunden_ausz 
    FROM stundauszahl
    WHERE
      sa_minr = _minr
      AND sa_date BETWEEN _date_start AND _date_end
      AND sa_type = 'hours';

    -- Ermittlung der Optimierungslohnart und Zeitbereiche
    SELECT int_start,      int_end,      loanr
    INTO   la_opt_1_start, la_opt_1_end, la_opt
    FROM TPersonal.bdep__lohnarten__optimierung__lohnart__get(1);
    
    SELECT int_start,      int_end      
    INTO   la_opt_2_start, la_opt_2_end
    FROM TPersonal.bdep__lohnarten__optimierung__lohnart__get(2);
    
    -- Einstiegsbedingung der Optimierung: IST-Stunden > SOLL-Stunden
    -- Nur dann können Mitarbeitende ihre Überstunden von STEUERPFLICHTIGEN Zuschlägen zu STEUERFREIEN verbessern 
    IF stukosaldo > 0 THEN          

        -- Über alle Lohnarzeiten im Zeitraum: an Tage mit Überstunden und in relevanten Optimierungszeiträumen
        -- und Berücksichtigung der Priorität der Lohnarten ~ je niedriger die prio, desto niedriger der Zuschlag
        -- bewusst vollständiges Statement für einfaches Debugging
        FOR rec_tmp_tab IN 
            SELECT
              loa_prio, 
              sum(tmp_tab.sum) over (partition by werktag) AS loa_sum_wt, -- Gesamtzeit der Lohnart am Werktag
              tmp_tab.*
              FROM (
                SELECT              
                  (tsystem.timestamps__intervals__intersect( 
                        timestamp_start => loa_anf,
                        timestamp_end   => loa_end,
                        interval_start  => la_opt_1_start, -- '20:00:00'::Time, Definition des Zeitraums vom Kunden
                        interval_end    => la_opt_1_end    --'24:00:00'::Time
                    )
                  ) AS do_opt1, --20_24,
                  (tsystem.timestamps__intervals__intersect( 
                        timestamp_start => loa_anf,
                        timestamp_end   => loa_end,
                        interval_start  => la_opt_2_start, -- '04:00:00'::Time, Definition des Zeitraums vom Kunden
                        interval_end    => la_opt_2_end    -- '06:00:00'::Time
                    )
                  ) AS do_opt2, --04_06,
                  tmp_time_tab.werktag,
                  tmp_time_tab.loa_nr, tmp_time_tab.loa_bez, tmp_time_tab.minr, 
                  tmp_time_tab.bd_anf_rund, tmp_time_tab.bd_end_rund,
                  tmp_time_tab.loa_anf, tmp_time_tab.loa_end, 
                  tmp_time_tab.min_exceed_sum, tmp_time_tab.sum         
                FROM tmp_time_tab
              ) AS tmp_tab
              JOIN lohnarten ON lohnarten.loa_nr = tmp_tab.loa_nr
            WHERE 
              min_exceed_sum > 0                         -- Überstunden haben stattgefunden
              AND  ( do_opt1 > 0 OR do_opt2 > 0)         -- relevante Optimierungszeiträume
           ORDER by loa_prio, bd_anf_rund, loa_anf       -- Priorität der Lohnart: niedrigsten Zuschläge zuerst
      
        LOOP
        -- Reduziere die Lohnartzuschläge an Tagen mit Überstunden nach der Reihenfolge der niedrigsten Zuschläge
        -- Damit für eine Überstunde Zuschläge nicht mehrfach gezahlt werden, müssen Überstunden, egal ob Auszahlung von Überstunden oder nicht, erst mal abgezogen werden
          -- und da es auch den Fall gibt, dass weniger Überstunden ausgezahlt werden, als in zuschlagsbehafteten Zeiträumen angefallen sind
          IF ( ( rec_tmp_tab.min_exceed_sum < rec_tmp_tab.sum )  -- Überstunden sind geringer als Lohnartzeit
               OR
               (     (rec_tmp_tab.min_exceed_sum > rec_tmp_tab.sum) -- Mehr Überstunden als Zeit in der Lohnart
                 AND (rec_tmp_tab.min_exceed_sum < rec_tmp_tab.loa_sum_wt ) -- Aber die Gesamtzeit der Lohnart am Werktag ist ausreichend
               )
             ) THEN
               ueberstunden_wt := rec_tmp_tab.min_exceed_sum;                     
          ELSE  
          --Gesamtzeit der Lohnart am Werktag ist NICHT ausreichend 
               ueberstunden_wt := rec_tmp_tab.sum;   -- dann nur die Zeit der Lohnart eintragen der Rest wird weiter unten von LA 1000 abgezogen                  
          END IF;
            
          -- Schreibe für jeden Werktag einen zusätzlich Eintrag in die temporäre Tabelle
            
          IF ( werktag_last IS NULL -- erster Eintrag
               OR werktag_last <> rec_tmp_tab.werktag -- die Überstunden nur einmal vom Werktag abziehen                    
             ) THEN
            PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
              rec_tmp_tab.werktag,
              _minr, 
              rec_tmp_tab.loa_nr, 
              ueberstunden_wt * -1,
              _debug
            ); 
            ---Raise notice 'optimierung__write: werktag= % loa_nr= % Abzug= %', rec_tmp_tab.werktag, rec_tmp_tab.loa_nr, ueberstunden_wt;    
            ueberstunden_wt_sum := ueberstunden_wt_sum + ueberstunden_wt; -- Bilde die Gesamtsumme des Abzugs von den relevanten Lohnarten 
          END IF;           
          werktag_last := rec_tmp_tab.werktag; 
          
          -- Fall berücksichtigen Freitzeitausgleich: Füge das zuviel abgezogene wieder dazu und verlassen den Loop
          IF (stukosaldo-ueberstunden_wt_sum) < 0 THEN
            -- Jetzt haben wir zuviel abgezogen: Stundenkonto-Saldo ist negativ
            ---Raise notice 'Skip denn stukosaldo-ueberstunden_wt_sum= %', stukosaldo-ueberstunden_wt_sum; 
            PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
              rec_tmp_tab.werktag,
              _minr, 
              rec_tmp_tab.loa_nr, 
              (stukosaldo-ueberstunden_wt_sum) * -1, -- positive Gutschrift erzeugen
              _debug
            );
            ---Raise notice 'optimierung__write: werktag= % loa_nr= % ueberstunden_wt_sum= % Plus= %', rec_tmp_tab.werktag, rec_tmp_tab.loa_nr, ueberstunden_wt_sum, (stukosaldo-ueberstunden_wt_sum) * -1; 
           
            --Jetzt den Wert von ueberstunden_wt_sum für weitere Verwendung wieder korrigieren
            ueberstunden_wt_sum := stukosaldo;
            ---Raise notice 'optimierung__write: Korrektur von ueberstunden_wt_sum = %', ueberstunden_wt_sum;
            
            Exit; --Hier den Loop verlassen
          END IF;                            

        END LOOP;
        
        -- Ab hier müssen die Lohnarten, die in Personalentgeltgruppe definiert sind, berücksichtigt werden ~ TPersonal.lohnarten__entgeltgruppe__minr__check    
        -- 2A Wenn keine Überstundenauszahlung, Dann schaue ob es noch mehr Überstunden als die bereits abgezogenen gibt
        IF ( ueberstunden_ausz IS NULL OR ueberstunden_ausz = 0 )  THEN
            ---Raise notice '2A keine Überstundenauszahlung';
            
            -- Es sind noch nicht alle Überstunden reduziert, diese werden pauschal von Lohnart 1000 abgezogen ~ Vereinbarung mit dem Kunden
            IF ( ueberstunden_wt_sum < stukosaldo 
                 AND TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, 1000)
               ) THEN 
              ---Raise notice '2A ueberstunden_wt_sum= % ', ueberstunden_wt_sum;
              PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                coalesce( _date_end, _date_start ),
                _minr, 
                1000, 
                (stukosaldo - ueberstunden_wt_sum) * -1,
                _debug
              ); 
              ---Raise notice 'optimierung__write: werktag= % Abzug= %', coalesce( _date_end, _date_start ), (stukosaldo - ueberstunden_wt_sum) * -1;
            END IF; 
        
        -- 2B Wenn Überstundenauszahlung, Dann werden folgende Lohnart-Salden erzeugt ~ 3 Fälle sind zu beachten
          -- 1) stundauszahl mehr als Überstunden, mehr als Abzug / 2) stundauszahl weniger als Überstunden, mehr als Abzug / 3) stundauszahl weniger als Überstunden, weniger als Abzug
          -- Fall 1) exemplarisch, andere nach ähnlicher Logik
            -- WENN Auszahlung > Überstunden des Monats UND Auszahlung > Summe Lohnartabzüge in den relevanten Zeiträumen        
            -- 2B1 Überstundenauszahlung 1100         : Überstunden des Monats ~ stukosaldo
            -- 2B2 Überstundenzuschlag 25% stpfl 1200 : Stundenauszahlung - Summe Lohnartabzüge in den relevanten Zeiträumen ~ wenn > 0
            -- 2B3 Früh/Spät 25% stfr 1530            : Summe Lohnartabzüge in den relevanten Zeiträumen
            -- 2B4 Einmalbezug 3890                   : Stundenauszahlung - Überstunden des Monats ~ wenn > 0  
        ELSE 
            ---Raise notice '2B Überstundenauszahlung';  
            
            -- 2B1 Überstundenauszahlung 1100
            IF ueberstunden_ausz < stukosaldo THEN
                sum := ueberstunden_ausz; -- kann maximal beantragten Auszahlung entsprechen
            ELSE 
                sum:= stukosaldo;         -- entspricht den Überstunden   
            END IF;
            IF TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, 1100) THEN
                PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                  coalesce( _date_end, _date_start ),
                  _minr, 
                  1100, 
                  sum,
                  _debug
                ); 
            END IF;
            
            -- 2B2 Überstundenzuschlag 25% stpfl 1200  
            IF ( (ueberstunden_ausz - ueberstunden_wt_sum) > 0 
                 AND TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, 1200) 
               ) THEN
               PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                 coalesce( _date_end, _date_start ),
                 _minr, 
                 1200, 
                 ueberstunden_ausz - ueberstunden_wt_sum, --Stundenauszahlung - Summe Lohnartabzüge in den relevanten Zeiträumen 
                 _debug
               ); 
               ---Raise notice '2B LA 1200 ueberstunden_wt_sum= %', ueberstunden_wt_sum;
            END IF; 
            
            -- 2B3 Früh/Spät 25% stfr 1530
            IF ueberstunden_ausz < ueberstunden_wt_sum THEN
                sum := ueberstunden_ausz;  -- kann maximal beantragten Auszahlung entsprechen
            ELSE 
                sum:= ueberstunden_wt_sum;  -- entspricht Summe Lohnartabzüge in den relevanten Zeiträumen   
            END IF;            
            IF  TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, la_opt)  -- la_opt 1530
                AND sum > 0 THEN  -- Schreiben von 0h soll vermieden werden
                  PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                    coalesce( _date_end, _date_start ),
                    _minr, 
                    la_opt, --1530, 
                    sum,
                    _debug
                ); 
            END IF;
              
            -- 2B4 Einmalbezug 3890 
            IF ( (ueberstunden_ausz - stukosaldo) > 0 
                 AND TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, 3890) 
               ) THEN
                PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                  coalesce( _date_end, _date_start ),
                  _minr, 
                  3890, 
                  ueberstunden_ausz - stukosaldo, -- Stundenauszahlung - Überstunden des Monats ~ wenn > 0
                  _debug
                );  
            END IF;
            
            -- Es sind noch nicht alle Überstunden reduziert, diese werden pauschal von Lohnart 1000 abgezogen ~ Vereinbarung mit dem Kunden          
            IF ( (ueberstunden_wt_sum + ueberstunden_ausz)  < stukosaldo 
                 AND TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, 1000)
               ) THEN 
              ---Raise notice '2B ueberstunden_wt_sum + ueberstunden_ausz= %; stukosaldo= % ', ueberstunden_wt_sum+ueberstunden_ausz, stukosaldo;
              PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                coalesce( _date_end, _date_start ),
                _minr, 
                1000, 
                (stukosaldo - (ueberstunden_wt_sum + ueberstunden_ausz)) * -1,
                _debug
              ); 
              Raise notice 'optimierung__write: werktag= % Abzug= %', coalesce( _date_end, _date_start ), (stukosaldo - (ueberstunden_wt_sum + ueberstunden_ausz)) * -1; 
            END IF;

        END IF;   
    
    ELSE 
      -- Sollzeit nicht erreicht (stukosaldo < 0) und Überstundenauszahlung, dann wird nichts optimiert, aber die Überstunden ausgezahlt 
      IF ueberstunden_ausz > 0 THEN
          ---Raise notice '2C Überstundenauszahlung aber Sollzeit nicht erreicht'; 
          
          -- 2C1 Überstundenzuschlag 25% stpfl 1200  
          IF ( TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, 1200) ) THEN
              PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                coalesce( _date_end, _date_start ),
                _minr, 
                1200, 
                ueberstunden_ausz, --die komplette Auszahlung 
                _debug
              ); 
          END IF; 
          
          -- 2C2 Einmalbezug 3890 
          IF ( TPersonal.lohnarten__entgeltgruppe__minr__check(_minr, 3890) ) THEN
              PERFORM tpersonal.bdep__lohnarten__optimierung__write( 
                coalesce( _date_end, _date_start ),
                _minr, 
                3890, 
                ueberstunden_ausz, --die komplette Auszahlung
                _debug
              );  
          END IF;
          ---Raise notice '2C LA 1200 und 3890 ueberstunden_ausz= %', ueberstunden_ausz;
      
      END IF;
      
    END IF;   

    RETURN;
  END $$ LANGUAGE plpgsql;
--
